<?php

/**
 * Code related to the settings-webinfo.php interface.
 *
 * PHP version 5
 *
 * @category   Library
 * @package    Sucuri
 * @subpackage SucuriScanner
 * @author     Daniel Cid <dcid@sucuri.net>
 * @copyright  2010-2018 Sucuri Inc.
 * @license    https://www.gnu.org/licenses/gpl-2.0.txt GPL2
 * @link       https://wordpress.org/plugins/sucuri-scanner
 */

if (!defined('SUCURISCAN_INIT') || SUCURISCAN_INIT !== true) {
    if (!headers_sent()) {
        /* Report invalid access if possible. */
        header('HTTP/1.1 403 Forbidden');
    }
    exit(1);
}

/**
 * Gather information from the server, database engine, and PHP interpreter.
 *
 * @return string HTML for the system info page.
 */
function sucuriscan_settings_webinfo_details()
{
    $params = array();
    $info_vars = array(
        'Plugin_version' => SUCURISCAN_VERSION,
        'Last_filesystem_scan' => SucuriScanFSScanner::getFilesystemRuntime(true),
        'Datetime_and_Timezone' => '',
        'Operating_system' => sprintf('%s (%d Bit)', PHP_OS, PHP_INT_SIZE * 8),
        'Server' => 'unknown',
        'WordPress_debug' => __('not active', 'sucuri-scanner'),
        'Memory_usage' => __('unknown', 'sucuri-scanner'),
        'PHP_version' => PHP_VERSION,
    );

    $params['ServerInfo.Variables'] = '';
    $info_vars['Datetime_and_Timezone'] = sprintf(
        '%s (%s)',
        SucuriScan::datetime(),
        SucuriScanOption::getOption(':timezone')
    );

    if (defined('WP_DEBUG') && WP_DEBUG) {
        $info_vars['WordPress_debug'] = __('active', 'sucuri-scanner');
    }

    if (function_exists('memory_get_usage')) {
        $info_vars['Memory_usage'] = round(memory_get_usage() / 1024 / 1024, 2).' MB';
    }

    if (isset($_SERVER['SERVER_SOFTWARE'])) {
        $info_vars['Server'] = $_SERVER['SERVER_SOFTWARE'];
    }

    /* PHP INI Settings */
    $field_names = array(
        'safe_mode',
        'expose_php',
        'allow_url_fopen',
        'memory_limit',
        'upload_max_filesize',
        'post_max_size',
        'max_execution_time',
        'max_input_time',
    );

    foreach ($field_names as $php_flag) {
        $php_flag_value = SucuriScan::iniGet($php_flag);
        $php_flag_name = 'PHP_' . $php_flag;
        $info_vars[$php_flag_name] = $php_flag_value ? $php_flag_value : 'N/A';
    }

    /* PHP constants */
    $constants = array(
        'ABSPATH',
        // 'SUCURISCAN',
        // 'SUCURISCAN_INIT',
        'SUCURISCAN_API_URL',
        'SUCURI_DATA_STORAGE',
        // 'SUCURISCAN_ADMIN_INIT',
        'SUCURISCAN_GET_PLUGINS_LIFETIME',
        'SUCURISCAN_THROW_EXCEPTIONS',
    );

    foreach ($constants as $name) {
        $info_vars[$name] = defined($name) ? constant($name) : '--';
    }

    foreach ($info_vars as $var_name => $var_value) {
        $var_name = str_replace('_', "\x20", $var_name);

        $params['ServerInfo.Variables'] .= SucuriScanTemplate::getSnippet(
            'settings-webinfo-details',
            array(
                'ServerInfo.Title' => $var_name,
                'ServerInfo.Value' => $var_value,
            )
        );
    }

    return SucuriScanTemplate::getSection('settings-webinfo-details', $params);
}

/**
 * Find the main htaccess file for the site and check whether the rules of the
 * main htaccess file of the site are the default rules generated by WordPress.
 *
 * @return string The HTML code displaying the information about the HTAccess rules.
 */
function sucuriscan_settings_webinfo_htaccess()
{

    $htaccess = SucuriScan::getHtaccessPath();
    $params = array(
        'HTAccess.Content' => '',
        'HTAccess.TextareaVisible' => 'hidden',
        'HTAccess.StandardVisible' => 'hidden',
        'HTAccess.NotFoundVisible' => 'hidden',
        'HTAccess.NotApache' => 'hidden',
        'HTAccess.FoundVisible' => 'hidden',
        'HTAccess.Fpath' => 'unknown',
    );

    // If it's not Apache, do not based the analysis on htaccess file.
    if (SucuriScan::isNginxServer() || SucuriScan::isIISServer()) {
        $params['HTAccess.NotApache'] = 'visible';
    } elseif ($htaccess) {
        $rules = SucuriScanFileInfo::fileContent($htaccess);

        $params['HTAccess.TextareaVisible'] = 'visible';
        $params['HTAccess.Content'] = $rules;
        $params['HTAccess.Fpath'] = $htaccess;
        $params['HTAccess.FoundVisible'] = 'visible';

        if (sucuriscan_htaccess_is_standard($rules)) {
            $params['HTAccess.StandardVisible'] = 'visible';
        }
    } else {
        $params['HTAccess.NotFoundVisible'] = 'visible';
    }

    return SucuriScanTemplate::getSection('settings-webinfo-htaccess', $params);
}

/**
 * Check if the standard rules for a normal WordPress installation (not network
 * based) are inside the main htaccess file. This only applies to websites that
 * have permalinks enabled, or 3rd-party plugins that require custom rules
 * (generally based on mod_deflate) to compress and/or generate static files for
 * cache.
 *
 * @param string $rules Content of the main htaccess file.
 * @return bool True if the htaccess has the standard rules, false otherwise.
 */
function sucuriscan_htaccess_is_standard($rules = '')
{
    if (!$rules) {
        $htaccess = SucuriScan::getHtaccessPath();

        if ($htaccess) {
            $rules = SucuriScanFileInfo::fileContent($htaccess);
        }
    }

    if (class_exists('WP_Rewrite') && is_string($rules) && !empty($rules)) {
        $rewrite = new WP_Rewrite();
        $standard = $rewrite->mod_rewrite_rules();

        if (!empty($standard)) {
            return (bool) (strpos($rules, $standard) !== false);
        }
    }

    return false;
}
